// ccc_control - implements lfo with amplitude control and skew, hfo with amplitude control, and offset
// lookup table used for sine generation, log conversion for pot controls, and skew calculations (1/x)
// uses all ADCs, sampled with timer1, 64x oversampling, rolling average.
// uses Timer1 PWM.  16b, Phase Correct, 31.25kHz.

#include <MiniArDSP.h>

#define PWM_FREQ 0x00FF // pwm frequency - see table
#define PWM_MODE 0 // Fast (1) or Phase Correct (0)
#define PWM_QTY 2 // number of pwms, either 1 or 2

PROGMEM  prog_int16_t sinewave[]  = { // this file is stored in MiniArDSP and is a 1024 value sinewave lookup table of signed 16bit integers
  #include <sinetable.inc>
};

PROGMEM  prog_uint16_t logtable4[]  = { // this file is stored in MiniArDSP and is a 1024 value log table: 30*2^(5.8*in/1024)
  #include <logtable4.inc>
};

PROGMEM  prog_uint16_t logtable3[]  = { // this file is stored in MiniArDSP and is a 1024 value log table: 192*2^(7.5*x/1024)
  #include <logtable3.inc>
};

PROGMEM  prog_uint8_t skewtable1[]  = { // this file is stored in MiniArDSP and is a 256 value 1/(1+x) table, 8b values
  #include <skew_lookup1.inc>
};

PROGMEM  prog_uint16_t skewtable2[]  = { // this file is stored in MiniArDSP and is a 256 value 1/(1-x) log table, 16b values
  #include <skew_lookup2.inc>
};

byte skew_state = 0;
unsigned long lfo_speed; // lookup table value location
unsigned long hfo_speed; // lookup table value location

// potentiometer values
unsigned int pot_values[6] = {0};

// lowpass filter storage for potentiometer values
unsigned int past_values[6][64] = {0};
byte pot_sample = 0; // which sample are we at
byte pot_location = 0;  // which pot are we sampling

void setup() {
  
  // debug bits
  //DDRD = 0x04; // set PORTD2 to output
  
  // setup ADC
  ADMUX = 0x40; // right adjust, adc0, internal vcc
  ADCSRA = 0xe5; // turn on adc, ck/32, auto trigger
  ADCSRB =0x07; // t1 capture for trigger
  DIDR0 = 0x3f; // turn off digital inputs for adc0:5
  
  // setup PWM
  TCCR1A = (((PWM_QTY - 1) << 5) | 0x80 | (PWM_MODE << 1)); // 
  TCCR1B = ((PWM_MODE << 3) | 0x11); // ck/1
  TIMSK1 = 0x20; // interrupt on capture interrupt
  ICR1H = (PWM_FREQ >> 8);
  ICR1L = (PWM_FREQ & 0xff);
  DDRB |= ((PWM_QTY << 1) | 0x02); // turn on outputs
  
  TIMSK0 = 0; // turn of t0 - no delay() or millis()
  sei(); // turn on interrupts - not really necessary with arduino
}

void loop() {
  while(1); // gets rid of jitter
  // nothing happens up here.  if you want to put code up here
  // get rid of the ISR_NAKED and the reti(); below
}

ISR(TIMER1_CAPT_vect, ISR_NAKED) { // ISR_NAKED is used to save clock cycles, but prohibits code in the loop() section.
  // debug
  //PORTD = 0x04;
  
  // get ADC data
  byte temp1 = ADCL; // you need to fetch the low byte first
  byte temp2 = ADCH; // yes it needs to be done this way
  unsigned int input = ((temp2 << 8) | temp1); // make an unsigned 16b value
  
  // lowpass ADC data and store in the appropriate register (64x oversample and moving average)
  temp1 = (ADMUX & 0x07); // fetch current mux postion - there is an off by one issue here compared to actual pins due to where this happens compared to the trigger
  pot_location = temp1;  // store for lowpass position
  pot_values[pot_location] += input - past_values[pot_location][pot_sample]; // do moving average 
  past_values[pot_location][pot_sample] = input; // store new value
  temp1++; // increment
  if (temp1 > 5) { // check if weve done them all
    temp1 = 0; // reset counter
    pot_sample++; // increment counter
    pot_sample &= 0x3f; // roll over counter at 63
  }
  ADMUX = (temp1 | 0x40); // reset mux to next value
  
  // calculate lfo
  input = pgm_read_word_near(logtable3 + (pot_values[1] >> 6));  // get speed
  temp1 = (pot_values[0] >> 8);
  if (skew_state) { // slow velocity ramp
    byte temp5 = pgm_read_byte_near(skewtable1 + temp1);
    unsigned int result;
    MultiU16X8toH16(result, input, temp5);
    lfo_speed += result;
    temp1 = (((lfo_speed >> 24) + 0x01) & 0x02);
    if (temp1 == 0) skew_state = 0;
  }
  else {  // fast velocity ramp
    unsigned int temp6 = pgm_read_word_near(skewtable2 + temp1);
    unsigned long result2;
    MultiU16X16to32(result2, input, temp6);
    lfo_speed += (result2 >> 8);
    temp1 = (((lfo_speed >> 24) + 0x01) & 0x02);
    if (temp1) {
      lfo_speed = 0x01000000; // set to top of count to prevent overflow
      skew_state = 1;
    }
  }
  signed int temp3 = pgm_read_word_near(sinewave + ((lfo_speed >> 16) & 0x03ff));
  signed int lfo;
  MultiSU16X16toH16(lfo, temp3, pot_values[2]);
  
  // calculate hfo
  input = pgm_read_word_near(logtable4 + (pot_values[4] >> 6));
  hfo_speed += input;
  temp3 = pgm_read_word_near(sinewave + ((hfo_speed >> 8) & 0x03ff));
  signed int hfo;
  MultiSU16X16toH16(hfo, temp3, pot_values[5]);
  
  //mix values
  signed long mix = pot_values[3];
  mix += lfo;
  mix += hfo;
  
  // truncate and convert to unsigned
  unsigned int temp4;
  if (mix < 0) temp4 = 0;
  else if (mix & 0xffff0000) temp4 = 0xffff;
  else temp4 = mix;
  
  // output data
  OCR1BL = temp4 >> 8; // convert to unsigned and output top byte
  OCR1AL = temp4; // output the bottom byte
  
  // debug
  //PORTD = 0x00;
  
  reti(); // return from interrupt - required because of ISR_NAKED
}
