;vamp2.asm
;guest openmusiclabs 12.9.12
;MIDIvampire-II firmware
;
;this is the main file for the midi synth.  it sets up the micro, and takes
;in midi data, and modifies parameters accordingly.  all of the voicing and
;modulation is done in the interrupts, for which there are seperate files.
;
;basic operation:
;
;1. there is no midi stack, data is handled as it arrives.  so there must be
;enough free time leftover after interrupt handling to do midi.  if you get
;more than a byte or 2 backlogged, you will get buffer over-runs.  the usart
;is 3 bytes deep, max.
;
;2. there is a 31.25kHz sample clock that processs the voicing data.  it must
;complete its task in under ~200 clock cycles, or there wont be time for midi
;and the other interrupts.  this has a 1 sample buffer, so 1 interrupt can be
;missed.
;
;3. there is a 244Hz interrupt that does attack, decay.  another 244Hz
;interrupt 90 degrees out of phase does frequency sweep.  2 more interrupts
;45 degrees out of phase do the fm (effectively at 488Hz).
;
;
;MIDI implementation
;
;note-on values of 32,33,34,35 turn on the 4 voices. velocity is ignored.
;note-off is ignored
;
;cc #	parameter (voice)
;----	---------------
; 12	attack (1)
; 13	decay (1)
; 14	release (1)
; 15	pitch (1)
; 16	frequency sweep (1)
; 17	bit-crush (1)
; 18	voice table (1)
; 19	table boundary (1)
;
; 20	attack (1)
; 21	decay (1)
; 22	release (1)
; 23	pitch (2)
; 24	frequency sweep (2)
; 25	bit-crush (2)
; 26	fm depth (2)
; 27	fm frequency (2)
; 28	voice (2)
;
; 75	attack (3)
; 76	decay (3)
; 77	release (3)
; 78	cut-off (3)
; 79	resonance start (3)
; 80	resonance stop (3)
; 81	resonance decay (3)
;
; 82	attack (4)
; 83	decay (4)
; 84	release (4)
; 85	pitch (4)
; 86	frequency sweep (4)
; 87	bit-crush (4)
; 88	fm depth (4)
; 89	fm frequency (4)
; 90	voice (4)

.include "m328pdef.inc" ; definitions file

; fuse settings
;
; external 8MHz resonator (low power oscillator), slowest startup time
; isp enabled
; brownout @ 1.8V
; no boot reset
; no memory locks
; LOW = 0xFF
; HIGH = 0xD9
; EXTENDED = 0xFE

; pin usage
;
; pb0   = nc
; pb1:2 = pwm out, oc1a / oc1b, t1, output
; pb3:5 = spi - isp
; pb6:7 = ceramic resonator, 8MHz
; pc0:3 = midi channel select 0:3, input, pullup on
; pc4:5 = nc
; pc6   = reset - isp
; pd0   = midi rx, usart0, pullup on
; pd1:4 = nc
; pd5   = pwm out, oc0b, t0, output
; pd6   = pwm out, oc0a, t0, output
; adc6  = nc
; adc7  = nc
; aref  = nc

; register usage
;
; r0  = multiply result lsb - do not use in main (not backed up)
; r1  = multiply result msb - do not use in main (not backed up)
; r2  = voice 1 pointer lsb
; r3  = voice 1 pointer msb
; r4  = voice 2 pointer lsb
; r5  = voice 2 pointer msb
; r6  = voice 3 pointer lsb - noise source
; r7  = voice 3 pointer msb
; r8  = voice 4 pointer lsb
; r9  = voice 4 pointer msb
; r10 = null register
; r11 = midi state (b7=data,b1=control,b0=note_on)
; r12 = voice 1 amplitude
; r13 = voice 2 amplitude
; r14 = voice 3 amplitude
; r15 = voice 4 amplitude
; r16 = main general swap register - main only
; r17 = main general swap register - main only
; r18 = interrupt general swap register - interrupts only
; r19 = interrupt general swap register - interrupts only
; r20 = interrupt general swap register - interrupts only
; r21 = interrupt general swap register - interrupts only
; r22 = current voice register - voice2
; r23 = ramp
; r24 = current voice register - voice4
; r25 = main general swap register - main only
; r26 = delay table pointer lsb - synth-engine only
; r27 = delay table pointer msb - synth-engine only
; r28 = delay table pointer lsb - synth-engine only
; r29 = delay table pointer msb - synth-engine only
; r30 = lookup table address lsb - backed up in interrupts
; r31 = lookup table address msb - backed up in interrupts
; gpior0 = pwm buffer1
; gpior1 = pwm buffer2
; gpior2 = pwm buffer3
; eedr = pwm buffer4
; eearl = cymbal sample pointer extra low byte


; sram usage
;
; $0100 - $07ff = delay memory
; $0800 - $08ef = variable storage
; $08f0 - $08ff = stack

.include "vamp2_defs.inc" ; definitions file with sram usage


.org $0000
rjmp init ; initialization
.org OC2Aaddr
rjmp t2int_a ; sweep handler
.org OC2Baddr
rjmp t2int_b ; fm handler
.org OVF2addr
rjmp t2int ; envelope handler
.org OVF0addr
rjmp t0int ; sample playback handler

init: ; stuff done only on reset

;move stack pointer to end of sram
ldi r16,high(ramend)
out sph,r16
ldi r16,low(ramend)
out spl,r16

;setup power reduction registers
ldi r16,(1<<prspi)|(1<<pradc)|(1<<prtwi)
sts prr,r16 ; turn off twi,spi,adc

;initialize io ports
ldi r16,(1<<ddd6)|(1<<ddd5) ; set pwms on t0 to output
out ddrd,r16
ldi r16,(1<<ddb2)|(1<<ddb1) ; set pwms on t1 to output
out ddrb,r16
sbi portd,pd0 ; turn on pullups for usart0 rx
ldi r16,(1<<ddc3)|(1<<ddc2)|(1<<ddc1)|(1<<ddc0) ; turn on pullups for midi channel switches
out portc,r16

;setup t0 for audio generation
;8b fast pwm, clear output on match, use both oca and ocb, ck/1 (31.25khz)
ldi r16,(1<<com0a1)|(0<<com0a0)|(1<<com0b1)|(0<<com0b0)|(1<<wgm01)|(1<<wgm00)
out tccr0a,r16
ldi r16,(0<<wgm02)|(1<<cs00)
out tccr0b,r16
ldi r16,(1<<toie0) ; turn on overflow interrupt
sts timsk0,r16

;setup t1 for audio generation
;8b fast pwm, clear output on match, use both oca and ocb, ck/1 (31.25khz)
ldi r16,(1<<com1a1)|(1<<com1b1)|(0<<wgm11)|(1<<wgm10)
sts tccr1a,r16
ldi r16,(0<<wgm13)|(1<<wgm12)|(1<<cs10)
sts tccr1b,r16 ; set to phase correct pwm mode, top = ocr1a, ck = Fcpu/64
ldi r16,$00
sts tccr1c,r16
sts ocr1ah,r16 ; initialize pwm output high bytes
sts ocr1al,r16 ; the high bytes must never be altered in later code
sts ocr1bh,r16
sts ocr1bl,r16
sts timsk1,r16 ; make sure all interrupts are off

;sync t0 and t1
ldi r16,(1<<tsm)|(1<<psrsync) ; halt prescaler
sts gtccr,r16 ; not sure if this works for prescaler of 1
clr r16
sts tcnt1h,r16
sts tcnt1l,r16
out tcnt0,r16
sts gtccr,r16 ; restart prescaler

;setup t2 for fm and envelopes
;phase correct, ocra top, ck/64
ldi r16,(0<<wgm21)|(1<<wgm20)
sts tccr2a,r16
ldi r16,(1<<wgm22)|(4<<cs20)
sts tccr2b,r16
ldi r16,$ff ; set top
sts ocr2a,r16
ldi r16,$7f ; set middle interrupt
sts ocr2b,r16
ldi r16,(1<<ocie2a)|(1<<ocie2b)|(1<<toie2) ; enable all interrupts
sts timsk2,r16

;setup usart0 for midi recieve
;8 data bits, 1 stop bit, no parity, 31.25kbps
ldi r16,(1<<rxen0)
sts ucsr0b,r16
ldi r16,(1<<ucsz01)|(1<<ucsz00)
sts ucsr0c,r16
ldi r16,0x0f ; 8MHz/31.25kHz/16 - 1
sts ubrr0l, r16

;fetch midi address
in r16,pinc
andi r16,$0f
sts midi_channel,r16

;initialize fixed purpose registers (except r22 and r24 - done in voices)
ldi r16,$de ; initialize random number generator
mov r6,r16
ldi r16,$ad
mov r7,r16
clr r10 ; clear the null register
clr r11 ; set midi state to waiting
clr r12 ; set all amplitudes to 0
clr r13
clr r14
clr r15
clr r23 ; shut off all voices - ramp
ldi r29,$01 ; prep delay pointer
clr r28

;initialize general use variables
clr r16
sts last_byte,r16 ; reset midi recieve buffer

;initialize voice1
ldi r16,$ff
sts attack1,r16
ldi r16,$f0
sts decay1,r16
ldi r16,$02
sts freq1_start_l,r16
sts freq1_start_h,r16
ldi r16,$01
sts sustain1,r16
ldi r16,$00
sts sweep1_l,r16
sts sweep1_h,r16
ldi r16,$ff
sts mask1,r16
ldi r16,$0f
sts voice1_mask,r16
ldi r16,$50
sts voice1_boundary,r16

;initialize voice2
ldi r16,$ff
sts attack2,r16
ldi r16,$f0
sts decay2,r16
ldi r16,$3f
sts sustain2,r16
ldi r16,$95
sts freq2_start_l,r16
ldi r16,$00
sts freq2_start_h,r16
ldi r16,$f0
sts sweep2_l,r16
ldi r16,$ff
sts sweep2_h,r16
ldi r16,$f8
sts mask2,r16
ldi r16,$00
sts fm_mod2_l,r16
sts fm_mod2_h,r16
sts fm_mod2,r16
ldi r16,$ff
sts fm_freq2,r16
ldi r22,$18

;initialize voice3
ldi r16,$ff
sts attack3,r16
ldi r16,$ed
sts decay3,r16
ldi r16,$16
sts sustain3,r16
ldi r16,$91
sts cutoff_l,r16
ldi r16,$04
sts cutoff_h,r16
ldi r16,$ff
sts res_start,r16
ldi r16,$55
sts res_stop,r16
ldi r16,$67
sts sweep_res,r16
ldi r16,$ff
sts mask3,r16

;initialize voice4
ldi r16,$ff
sts attack4,r16
ldi r16,$e8
sts decay4,r16
ldi r16,$01
sts sustain4,r16
ldi r16,$20
sts freq4_start_l,r16
ldi r16,$0c
sts freq4_start_h,r16
ldi r16,$00
sts sweep4_l,r16
sts sweep4_h,r16
ldi r16,$ff
sts mask4,r16
ldi r16,$54
sts fm_mod4_l,r16
sts fm_mod4,r16
ldi r16,$00
sts fm_mod4_h,r16
ldi r16,$01
sts fm_freq4,r16
ldi r24,$40

;enable interrupts
ldi r16,$07
out tifr2,r16 ; clear any pending interrupts
out tifr0,r16 ; clear any pending interrupts
out tifr1,r16 ; clear any pending interrupts
sei ; turn on interrupts


main: ; non-interrupt code - process midi data

;check if data is in the rx buffer
lds r16,ucsr0a
sbrs r16,rxc0
rjmp main ; keep checking

;process incoming data
lds r25,udr0 ; store quickly in case another byte is arriving
sbrs r25,$07 ; check if command byte
rjmp data_byte
lds r16,midi_channel
ori r16,$90
cp r25,r16 ; now $90 + midi channel
breq note_on
ori r16,$b0
cp r25,r16 ; now $b0 + midi channel - this only works because of the order
breq control
cpi r25,$f8 ; check if a realtime message
brsh main ; keep same status if realtime message
clr r11 ; if none of the above, set midi state to 0
rjmp main ; go back to checking

note_on: ; set to note_on state

ldi r16,$01 ; set note_on bit, clear midi state and data bit
mov r11,r16
rjmp main

control: ; set to control state

ldi r16,$02 ; set control bit, clear midi state and data bit
mov r11,r16
rjmp main

data_byte: ; put data in the right place

sbrc r11,$07 ; check if second_byte in sequence
rjmp second_byte
mov r16,r11
andi r16,$03 ; check if null state
breq main ; skip if not one of the used states
sts last_byte,r25 ; backup data
ldi r16,$80
or r11,r16 ; set second_byte bit
rjmp main ; go back to waiting

second_byte: ; process data

ldi r16,$7f
and r11,r16 ; clear second_byte bit
sbrc r11,$00
rjmp note_on_process
sbrc r11,$01
rjmp control_process
rjmp main ; return to waiting

note_on_process: ; set notes

tst r25 ; check if note_off velocity (0)
breq main ; note offs are ignored
lds r16,last_byte ; fetch note number
cpi r16,$20
breq voice1_on ; turn off that voice if so
cpi r16,$21
breq voice2_on
cpi r16,$22
breq voice3_on
cpi r16,$23
brne main ; if no matches, do nothing
rjmp voice4_on

voice1_on: ; turn on voice1

lds r16,freq1_start_l
lds r17,freq1_start_h
cli ; block in case it gets modified in an interrupt
sts freq1_l,r16
sts freq1_h,r17
andi r23,$fc
ori r23,$01 ; set voice1 to attack
clr r2
clr r3 ; sync oscillator regardless of decay
sei
rjmp main ; go back to waiting

voice3_on: ; turn on voice3

lds r16,res_start
cli ; block in case it gets modified in an interrupt
sts resonance,r16
andi r23,$cf
ori r23,$10 ; set voice3 to attack
sei
rjmp main ; go back to waiting

voice2_on: ; turn on voice2

lds r16,freq2_start_l
lds r17,freq2_start_h
lds r25,fm_mod2
cli ; block in case it gets modified in an interrupt
sts freq2_l,r16
sts freq2_h,r17
sts fm_mod2_l,r25 ; reset fm
clr r16
sts fm_mod2_h,r16
inc r16
sts fm_timer2,r16 ; reset fm timer
mov r16,r23 ; check if voice currently on
andi r16,$0c
brne voice2_skip
clr r4 ; sync oscillator if not on
clr r5

voice2_skip:

andi r23,$f3
ori r23,$04 ; set voice1 to attack
sei
rjmp main ; go back to waiting

voice4_on: ; turn on voice4

lds r16,freq4_start_l
lds r17,freq4_start_h
lds r25,fm_mod4
cli ; block in case it gets modified in an interrupt
sts freq4_l,r16
sts freq4_h,r17
sts fm_mod4_l,r25 ; reset fm
clr r16
sts fm_mod4_h,r16
inc r16
sts fm_timer4,r16 ; reset fm timer
mov r16,r23 ; check if voice currently on
andi r16,$c0
brne voice4_skip
clr r8 ; sync oscillator if not on
clr r9

voice4_skip:

andi r23,$3f
ori r23,$40 ; set voice1 to attack
sei
rjmp main ; go back to waiting

control_process: ; check which controller is active

;check which controller is active
lds r16,last_byte
cpi r16,$4b ; check if controller bank 3 or 4
brsh control_process2
cpi r16,$14 ; check if controller 2
brlo control_process3
cpi r16,$1d ; check if out of range for controller 2
brsh control_process_done
rjmp controller2 ; do controller 2 functions

control_process2:

cpi r16,$52 ; check if controller 4
brsh control_process4
rjmp controller3

control_process3:

cpi r16,$0c ; check if out of range for controller 1
brsh controller1
rjmp main

control_process4:

cpi r16,$5b ; check if out of range for controller 4
brsh control_process_done
rjmp controller4

control_process_done:

rjmp main ; error - return to waiting

controller1:

mod11_1: ; attack

cpi r16,$0c
brne mod12_1
neg r25 ; reverse knob direction
subi r25,$80 ; give an offset
sbrc r25,$07 ; check for largest value
ldi r25,$ff ; set to max
sts attack1,r25 ; save new attack variable
rjmp main ; go back to waiting

mod12_1: ; decay

cpi r16,$0d
brne mod13_1
subi r25,$80 ; add in offset
sts decay1,r25 ; save new decay variable
rjmp main ; go back to waiting

mod13_1: ; sustain

cpi r16,$0e
brne mod14_1
neg r25 ; reverse knob direction
subi r25,$80 ; add offset
sts sustain1,r25
rjmp main

mod14_1: ; pitch

cpi r16,$0f
brne mod15_1
mov r30,r25
lsl r30 ; adjust for 2byte fetch
ldi r31,$0c
lpm r16,z+
lpm r17,z
sts freq1_start_l,r16
sts freq1_start_h,r17
cli
sts freq1_l,r16
sts freq1_h,r17
sei
rjmp main

mod15_1: ; frequency sweep

cpi r16,$10
brne mod16_1
clr r16
subi r25,$40
sbc r16,r10
sts sweep1_l,r25
sts sweep1_h,r16
rjmp main

mod16_1: ; bitcrush

cpi r16,$11
brne mod17_1
com r25
sts mask1,r25
rjmp main 

mod17_1: ; voice1 table

cpi r16,$12
brne mod18_1
lsr r25
lsr r25
andi r25,$10
subi r25,$b0
sts voice1_boundary,r25
rjmp main

mod18_1: ; voice1 table boundary

cpi r16,$13
brne mod_none_1
lsr r25
lsr r25
sts voice1_mask,r25
rjmp main

mod_none_1: ; not one of our states

rjmp main ; go back to waiting

controller2:

mod11_2: ; attack

cpi r16,$14
brne mod12_2
neg r25 ; reverse knob direction
subi r25,$80 ; give an offset
sbrc r25,$07 ; check for largest value
ldi r25,$ff ; set to max
sts attack2,r25 ; save new attack variable
rjmp main ; go back to waiting

mod12_2: ; decay

cpi r16,$15
brne mod13_2
subi r25,$80 ; add in offset
sts decay2,r25 ; save new decay variable
rjmp main ; go back to waiting

mod13_2: ; sustain

cpi r16,$16
brne mod14_2
neg r25 ; reverse knob direction
subi r25,$80 ; add offset
sts sustain2,r25
rjmp main

mod14_2: ; pitch

cpi r16,$17
brne mod15_2
mov r30,r25
lsl r30 ; adjust for 2byte fetch
ldi r31,$0c
lpm r16,z+
lpm r17,z
sts freq2_start_l,r16
sts freq2_start_h,r17
cli
sts freq2_l,r16
sts freq2_h,r17
sei
rjmp main

mod15_2: ; frequency sweep

cpi r16,$18
brne mod16_2
clr r16
subi r25,$40
sbc r16,r10
sts sweep2_l,r25
sts sweep2_h,r16
rjmp main

mod16_2: ; bitcrush

cpi r16,$19
brne mod17_2
com r25
sts mask2,r25
rjmp main

mod17_2: ; fm_mod

cpi r16,$1a
brne mod18_2
clr r16
lsl r25
cli
sts fm_mod2,r25
sts fm_mod2_l,r25
sts fm_mod2_h,r16
sei
rjmp main

mod18_2: ; fm_freq

cpi r16,$1b
brne mod19_2
com r25 ; reverse knob direction
subi r25,$7f ; add in offset
sts fm_freq2,r25
rjmp main

mod19_2: ; voice

cpi r16,$1c
brne mod_none_2
lsr r25 ; adjust for 8 options at lookup table boundary
andi r25,$38
subi r25,$f0
mov r22,r25 ; move to voice register
rjmp main

mod_none_2: ; if none of the above

rjmp main ;go back to waiting

controller3:

mod11_3: ; attack

cpi r16,$4b
brne mod12_3
neg r25 ; reverse knob direction
subi r25,$80 ; give an offset
sbrc r25,$07 ; check for largest value
ldi r25,$ff ; set to max
sts attack3,r25 ; save new attack variable
rjmp main ; go back to waiting

mod12_3: ; decay

cpi r16,$4c
brne mod13_3
subi r25,$80 ; add in offset
sts decay3,r25 ; save new decay variable
rjmp main ; go back to waiting

mod13_3: ; sustain

cpi r16,$4d
brne mod14_3
neg r25 ; reverse knob direction
subi r25,$80 ; add offset
sts sustain3,r25
rjmp main

mod14_3: ; cutoff

cpi r16,$4e
brne mod15_3
lsl r25 ; prep for 2 byte fetch
mov r30,r25
ldi r31,$0b
lpm r16,z+
lpm r17,z
sts cutoff_l,r16
sts cutoff_h,r17
rjmp main

mod15_3: ; resonance start

cpi r16,$4f
brne mod16_3
lsl r25
com r25 ; reverse knob direction
sts res_start,r25
sts resonance,r25 ; update resonace immediately
rjmp main

mod16_3: ; resonance stop

cpi r16,$50
brne mod17_3
lsl r25
com r25 ; reverse knob direction
sts res_stop,r25
rjmp main

mod17_3: ; resonance sweep

cpi r16,$51
brne mod_none_3
neg r25 ; reverse knob direction
subi r25,$80 ; give an offset
andi r25,$7f ; make the first value 0 for off
sts sweep_res,r25
rjmp main

mod_none_3: ; if none of the above

rjmp main ;go back to waiting

controller4:

mod11_4: ; attack

cpi r16,$52
brne mod12_4
neg r25 ; reverse knob direction
subi r25,$80 ; give an offset
sbrc r25,$07 ; check for largest value
ldi r25,$ff ; set to max
sts attack4,r25 ; save new attack variable
rjmp main ; go back to waiting

mod12_4: ; decay

cpi r16,$53
brne mod13_4
subi r25,$80 ; add in offset
sts decay4,r25 ; save new decay variable
rjmp main ; go back to waiting

mod13_4: ; sustain

cpi r16,$54
brne mod14_4
neg r25 ; reverse knob direction
subi r25,$80 ; add offset
sts sustain4,r25
rjmp main

mod14_4: ; pitch

cpi r16,$55
brne mod15_4
mov r30,r25
lsl r30 ; adjust for 2byte fetch
ldi r31,$0c
lpm r16,z+
lpm r17,z
sts freq4_start_l,r16
sts freq4_start_h,r17
cli
sts freq4_l,r16
sts freq4_h,r17
sei
rjmp main

mod15_4: ; frequency sweep

cpi r16,$56
brne mod16_4
clr r16
subi r25,$40
sbc r16,r10
sts sweep4_l,r25
sts sweep4_h,r16
rjmp main

mod16_4: ; bitcrush

cpi r16,$57
brne mod17_4
com r25
sts mask4,r25
rjmp main

mod17_4: ; fm_mod

cpi r16,$58
brne mod18_4
clr r16
lsl r25
cli
sts fm_mod4,r25
sts fm_mod4_l,r25
sts fm_mod4_h,r16
sei
rjmp main

mod18_4: ; fm_freq

cpi r16,$59
brne mod19_4
com r25 ; reverse knob direction
subi r25,$7f ; add offset
sts fm_freq4,r25
rjmp main

mod19_4: ; voice

cpi r16,$5a
brne mod_none_4
lsr r25 ; adjust for 8 options at lookup table boundary
andi r25,$38
subi r25,$f0
mov r24,r25 ; move to voice register
rjmp main

mod_none_4: ; if none of the above

rjmp main ;go back to waiting


t0int: ; do 32khz sound generation
.include "drum-synth.inc"

t2int: ; do envelopes here
.include "envelope-engine2.inc"

t2int_b: ; do frequency modulation
.include "fm_mod.inc"

t2int_a: ; do frequency sweep here
.include "sweeps.inc"

;.org $0500 ; lookup table for bitcrusher - not using it
;.include "bit_crush.inc"

.org $0580 ; delay table for filter
.include "delay_table.inc"

.org $0600 ; lookup table for note to frequency converter
.include "notelookup_assy.inc"

.org $0800 ; voice1 table
.include "sine_table_2k_signed.inc" ; 8 bit, 2048 sample, signed, sinewave

.org $0c00 ; voice2 table
.include "square.inc" ; 8 bit, 2048 sample

.org $1000 ; voice3 table
.include "eeh.inc"

.org $1400 ; voice4 table
.include "aay.inc" ; 8 bit, 2048 sample

.org $1800 ; voice5 table
.include "lid1.txt" ; 8 bit, 2048 sample

.org $1c00 ; voice6 table
.include "tube1.txt" ; 8 bit, 2048 sample

.org $2000 ; voice7 table
.include "pot1.txt" ; 8 bit, 2048 sample

.org $2400 ; voice8 table
.include "noise3.txt" ; 8 bit, 2048 sample

.org $2800 ; voice3 table1
.include "ohh2.txt" ;  8 bit, 4096 sample, signed, open hihat

.org $3000 ; voice3 table2
.include "cymbal_assy.inc" ; 8 bit, 8192 sample, signed, cymbal
