/*
variable_delay.pde
for MICrODEC
guest openmusiclabs 8.11.11
this program takes input from the left channel, and delays it
by a variable amount, set by the MOD1 knob.  it then presents
this delayed data on the left channel, with the non-delayed
data coming out of the right channel.  only the first 16b
address space of the external SRAM is used in this example.
these memwrite16 and memread16 functions are faster than
using the full 18b address space, and give 3s of delay in mono.
*/

// setup codec parameters
// must be done before #includes
// see readme file in libraries folder for explanations
#define SAMPLE_RATE 44 // 44.1kHz sample rate
#define MOD1 1 // use MOD1
#define MOD2 0 // dont use MOD2

// include necessary libraries
#include <Microdec.h>

// create data variables for audio transfer
// even though the function is mono, the codec requires stereo data
int left_in = 0; // in from codec (LINE_IN)
int right_in = 0;
int left_out = 0; // out to codec (HP_OUT)
int right_out = 0;

// create variable for MOD1 result
// it only has positive values -> unsigned
unsigned int mod1_value = 0;

// create a delay buffer in memory
#define SIZE 0xa456 // buffer size is limited by external SRAM size
unsigned int location = 0; // buffer location to read/write from
unsigned int boundary = 0; // end of buffer position

void setup() {
  // call this last if you are setting up other things
  Microdec_init(); // setup codec and microcontroller registers
}

void loop() {
  while (1); // reduces clock jitter
}

// timer1 interrupt routine - all data processed here
ISR(TIMER0_COMP_vect, ISR_NAKED) { // dont store any registers

  // &'s are necessary on data_in variables
  Microdec_data(&left_in, &right_in, left_out, right_out);
  
  // pass left input to right output
  right_out = left_in;

  //fetch data from buffer and put it into output register
  left_out = Microdec_memread16(location);
  // put new data in same location for maximal delay time
  Microdec_memwrite16(location++, left_in); // post increment location to go to next memory location
  // check if location has gotten bigger than buffer size
  if (location >= boundary) {
    location = 0; // reset location
  }

  // & is required before adc variable
  Microdec_mods(&mod1_value);

  // scale MOD1 value to match buffer size
  // note the use of the fancy fast math function
  // you can read about these in the readme file
  unsigned int scale = SIZE;
  MultiU16X16toH16(boundary, mod1_value, scale);

  reti(); // dont forget to return from the interrupt
}

