// Microdec.cpp
// guest openmusiclabs 8.11.11
// this has a few utility functions for talking i2c
// as the amtega3250p doesnt have and i2c hardware peripheral


#include "WProgram.h"
#include "Microdec.h"


// us delay timer (~1us per unit)
// sets the i2c clock rate
void mydelay(unsigned char t) {
  t <<= 1;
  while(t > 0) {
    asm volatile("nop");
    asm volatile("nop");
    asm volatile("nop");
    asm volatile("nop");
    asm volatile("nop");
    asm volatile("nop");
    t--;
  }
}


// i2c start condition
char i2cbb_start(void) {
  unsigned char k = PINE & (1<<PINE5);
  unsigned char l = PINE & (1<<PINE4);
  if (k == 0) { // check if data line released
    return -1; // end with failure if not released
  }
  else if (l == 0) { // check if clock line released
    return -2; // end with failure if not released
  }
  else { // send start condition
    DDRE |= (1<<DDE5);  // data low
    mydelay(10); // delay
    DDRE |= (1<<DDE4); // clock low
    mydelay(10); // delay
    return 1; // set state to success
  }
}


// i2c stop condition
void i2cbb_stop(void) {
  DDRE |= (1<<DDE5); // pull data low
  mydelay(10); // delay
  DDRE &= ~(1<<DDE4); // release clock line
  mydelay(10); // delay
  DDRE &= ~(1<<DDE5); // release data line
  mydelay(40); // delay to make sure a new data transfer doesnt occur too quickly
}


// i2c data send
char i2cbb_send(unsigned char data) {  // clock out data
  unsigned char state = 0; // initialize return state
  unsigned char i = 0x80;
  unsigned char k;
  do {
    k = data & i;
    if (k == 0) {
      DDRE |= (1<<DDE5); // data low
    }
    else {
      DDRE &= ~(1<<DDE5); // data high
    }
    mydelay(10);
    DDRE &= ~(1<<DDE4); // clock high
    mydelay(10);
    DDRE |= (1<<DDE4); // clock low
    i >>= 1;
  } while (i > 0);
  // check for ack
  DDRE &= ~(1<<DDE5); // release data line
  unsigned char d = 30; // initialize timeout  
  do {
    mydelay(10); // wait a bit    
    d--;
    if (d == 0) {
      state = 2; // set i2c state to nack
      break;
    }
    k = PINE & (1<<PINE5); // check for ack
  } while (k == 1); // keep checking till ack or timeout
  // clock the ack or nack
  DDRE &= ~(1<<DDE4); // clock high
  mydelay(10);
  DDRE |= (1<<DDE4); // clock low
  // make sure line is released
  d = 30; // set timeout
  do {
    mydelay(10);
    d--;
    if (d == 0) {
      state = 3; // set i2c state to no line release
      break;
    }
    k = PINE & (1<<PINE5);
  } while (k == 0);
  if (state > 1) { // send stop if failure
    i2cbb_stop();
  }
  else { // set state to success
    state = 1;
  }
  return state;
}

// full i2c protocol for 3 byte transfer
unsigned char i2cbb(unsigned char reg, unsigned char data) {
  if (i2cbb_start() != 1) { // send start condition
    return 2;
  }
  else if (i2cbb_send(ADDR) != 1) { // send address and write bit
    return 3;
  }
  else if (i2cbb_send(reg) != 1) { // send register to write to
    return 4;
  }
  else if (i2cbb_send(data) != 1) { // write data to register
    return 5;
  }
  else {
    i2cbb_stop(); // send stop condition
    return 1;
  }
}