// Microdec.h
// guest openmusiclabs 7.27.11
// this is the library file for programming your MICRoDEC with ARDUINO
// place this file in the libraries file of your Arduino sketches folder.
// e.g. 


#ifndef Microdec_h // include guard
#define Microdec_h

#include "WProgram.h"
#include <avr/pgmspace.h>
#include "mult16x16.h"
#include "mult16x8.h"
#include "mult32x16.h"

#define ADDR 0x34 // i2c address and write bit


// microdec can only handle these as timer0 is only 8bit
#ifndef SAMPLE_RATE
  #define SAMPLE_RATE 44
#elif (SAMPLE_RATE == 88)||(SAMPLE_RATE == 44)||(SAMPLE_RATE == 22)
#else
  #error SAMPLE_RATE value not defined
#endif

#ifndef ADCHPD
  #define ADCHPD 0
#elif (ADCHPD == 0)||(ADCHPD == 1)
#else
  #error ADCHPD value not defined
#endif

#ifndef MOD1
  #define MOD1 0
#elif (MOD1 == 0)||(MOD1 == 1)
#else
  #error MOD1 value not defined
#endif

#ifndef MOD2
  #define MOD2 0
#elif (MOD2 == 0)||(MOD2 == 1)
#else
  #error MOD2 value not defined
#endif

#ifndef HYST
  #define HYST 32
#elif (HYST >= 0)&&(HYST <= 255)
#else
  #error HYST value not defined
#endif

#ifndef MOD2INDEX
  #define MOD2INDEX 1
#elif (MOD2INDEX >= 1)&&(MOD2INDEX <= 255)
#else
  #error MOD2INDEX value not defined
#endif

#ifndef MOD2OVF
  #define MOD2OVF 1
#elif (MOD2OVF == 0)||(MOD2OVF == 1)
#else
  #error MOD2OVF value not defined
#endif

#ifndef MOD2START
  #define MOD2START 128
#elif (MOD2START >= 0)&&(MOD2START <= 255)
#else
  #error MOD2START value not defined
#endif

#ifndef LINVOL
  #define LINVOL 23
#elif (LINVOL >= 0) && (LINVOL <= 0x1f)
#else
  #error LINVOL value not defined
#endif

#ifndef RINVOL
  #define RINVOL 23
#elif (RINVOL >= 0) && (RINVOL <= 0x1f)
#else
  #error RINVOL value not defined
#endif

#ifndef LHPVOL
  #define LHPVOL 121
#elif (LHPVOL == 0) || ((LHPVOL >= 0x30) && (LHPVOL <= 0x7f))
#else
  #error LHPVOL value not defined
#endif

#ifndef RHPVOL
  #define RHPVOL 121
#elif (RHPVOL == 0) || ((RHPVOL >= 0x30) && (RHPVOL <= 0x7f))
#else
  #error RHPVOL value not defined
#endif

#ifndef MICBOOST
  #define MICBOOST 0
#elif (MICBOOST == 0)||(MICBOOST == 1)
#else
  #error MICBOOST value not defined
#endif

#ifndef MUTEMIC
  #define MUTEMIC 1
#elif (MUTEMIC == 0)||(MUTEMIC == 1)
#else
  #error MUTEMIC value not defined
#endif

#ifndef INSEL
  #define INSEL 0
#elif (INSEL == 0)||(INSEL == 1)
#else
  #error INSEL value not defined
#endif

#ifndef BYPASS
  #define BYPASS 0
#elif (BYPASS == 0)||(BYPASS == 1)
#else
  #error BYPASS value not defined
#endif

#ifndef DACSEL
  #define DACSEL 1
#elif (DACSEL == 0)||(DACSEL == 1)
#else
  #error DACSEL value not defined
#endif

#ifndef SIDETONE
  #define SIDETONE 0
#elif (SIDETONE == 0)||(SIDETONE == 1)
#else
  #error SIDETONE value not defined
#endif

#ifndef SIDEATT
  #define SIDEATT 0
#elif (SIDEATT >= 0)&&(SIDEATT <= 3)
#else
  #error SIDEATT value not defined
#endif

#ifndef OVERSAMPLE
  #define OVERSAMPLE 64
#elif (OVERSAMPLE == 1)||(OVERSAMPLE == 2)||(OVERSAMPLE == 4)||(OVERSAMPLE == 8)||(OVERSAMPLE == 16)||(OVERSAMPLE == 32)||(OVERSAMPLE == 64)
#else
  #error OVERSAMPLE value not defined
#endif


// setup variables for ADC
#if MOD1 == 0
  // do nothing
#elif MOD1 == 1
  unsigned char _i = OVERSAMPLE;
  unsigned int _mod1temp = 0x0000;
#endif

#if MOD2 == 0
  // do nothing
#elif MOD2 == 1
  unsigned char _k = 64*44/SAMPLE_RATE; // keeps it at 1.5ms
  unsigned char _mod2temp = MOD2START;
#endif


// the first section is i2c bit banging and setup
// as the usi interface is useless


// us delay timer (~1us per unit)
// sets the i2c clock rate
void mydelay(unsigned char t) {
  t <<= 1;
  while(t > 0) {
    asm volatile("nop");
    asm volatile("nop");
    asm volatile("nop");
    asm volatile("nop");
    asm volatile("nop");
    asm volatile("nop");
    t--;
  }
}


// i2c start condition
char i2cbb_start(void) {
  unsigned char k = PINE & (1<<PINE5);
  unsigned char l = PINE & (1<<PINE4);
  if (k == 0) { // check if data line released
    return -1; // end with failure if not released
  }
  else if (l == 0) { // check if clock line released
    return -2; // end with failure if not released
  }
  else { // send start condition
    DDRE |= (1<<DDE5);  // data low
    mydelay(10); // delay
    DDRE |= (1<<DDE4); // clock low
    mydelay(10); // delay
    return 1; // set state to success
  }
}


// i2c stop condition
void i2cbb_stop(void) {
  DDRE |= (1<<DDE5); // pull data low
  mydelay(10); // delay
  DDRE &= ~(1<<DDE4); // release clock line
  mydelay(10); // delay
  DDRE &= ~(1<<DDE5); // release data line
  mydelay(40); // delay to make sure a new data transfer doesnt occur too quickly
}


// i2c data send
char i2cbb_send(unsigned char data) {  // clock out data
  unsigned char state = 0; // initialize return state
  unsigned char i = 0x80;
  unsigned char k;
  do {
    k = data & i;
    if (k == 0) {
      DDRE |= (1<<DDE5); // data low
    }
    else {
      DDRE &= ~(1<<DDE5); // data high
    }
    mydelay(10);
    DDRE &= ~(1<<DDE4); // clock high
    mydelay(10);
    DDRE |= (1<<DDE4); // clock low
    i >>= 1;
  } while (i > 0);
  // check for ack
  DDRE &= ~(1<<DDE5); // release data line
  unsigned char d = 30; // initialize timeout  
  do {
    mydelay(10); // wait a bit    
    d--;
    if (d == 0) {
      state = 2; // set i2c state to nack
      break;
    }
    k = PINE & (1<<PINE5); // check for ack
  } while (k == 1); // keep checking till ack or timeout
  // clock the ack or nack
  DDRE &= ~(1<<DDE4); // clock high
  mydelay(10);
  DDRE |= (1<<DDE4); // clock low
  // make sure line is released
  d = 30; // set timeout
  do {
    mydelay(10);
    d--;
    if (d == 0) {
      state = 3; // set i2c state to no line release
      break;
    }
    k = PINE & (1<<PINE5);
  } while (k == 0);
  if (state > 1) { // send stop if failure
    i2cbb_stop();
  }
  else { // set state to success
    state = 1;
  }
  return state;
}

// full i2c protocol for 3 byte transfer
unsigned char i2cbb(unsigned char reg, unsigned char data) {
  if (i2cbb_start() != 1) { // send start condition
    return 2;
  }
  else if (i2cbb_send(ADDR) != 1) { // send address and write bit
    return 3;
  }
  else if (i2cbb_send(reg) != 1) { // send register to write to
    return 4;
  }
  else if (i2cbb_send(data) != 1) { // write data to register
    return 5;
  }
  else {
    i2cbb_stop(); // send stop condition
    return 1;
  }
}



static inline void Microdec_init(void) {

  // setup SRAM io lines
  DDRD = 0xff; // set portd as output for address lines
  DDRH = 0xff; // set porth as output for address lines
  PORTA = 0xff; // turn on pullups on porta for data lines
  PORTC = 0xff; // turn on pullups on portc for data lines
  DDRA = 0; // set porta as input for data lines
  DDRC = 0; // set portc as input for data lines
  DDRG |= 0x0f; // set portg sram control pins to output
  DDRB |= (1 << DDB7); // set oe control pin to output
  PORTB &= ~(1 << PORTB7); // set oe to low - redefined in SPI setup

  // setup spi peripheral
  DDRB |= ((1<<PB2)|(1<<PB1)|(1<<PB0)); // set ss,sck,mosi as output
  DDRB &= ~(1<<PB3); // set miso as input
  SPCR = 0x50; // set spi to master, mode 0
  SPSR = 0x01; // set spi to 2x (10MHz)

  // setup switch lines
  DDRJ &= ~(0x7c); // make sure switch lines are inputs
  PORTJ = 0x7c; // turn on pullups for rotary switch and pushbutton

  // setup i2c pins - make sure pullups are off
  PORTE &= ~((1<<PE5)|(1<<PE4)); // turn off pullups
  DDRE &= ~((1<<DDE5)|(1<<DDE4)); // set i2c lines to input

  // codec register setup using i2c
  // see wm8731 datasheet for other options (starting pg 46)
  // each register retries until success
  // if communication fails the device will hang
  while (i2cbb(0x0c, 0x00) != 1) { // power save registers -> all on
    delay(10);
  }
  while (i2cbb(0x0e, 0x03) != 1) { // digital data format -> 16b spi mode
    delay(10);
  }
  while (i2cbb(0x00, LINVOL) != 1) { // left input configure
    delay(10);
  }
  while (i2cbb(0x02, RINVOL) != 1) { // right input configure
    delay(10);
  }
  while (i2cbb(0x04, LHPVOL) != 1) { // left headphone configure
    delay(10);
  }
  while (i2cbb(0x06, RHPVOL) != 1) { // right headphone configure
    delay(10);
  }
  while (i2cbb(0x0a, ADCHPD) != 1) { // digital pathway configure
    delay(10);
  }
  while (i2cbb(0x08, (SIDEATT << 6)|(SIDETONE << 5)|(DACSEL << 4)|(BYPASS << 3)|(INSEL << 2)|(MUTEMIC << 1)|(MICBOOST << 0)) != 1) { // analog pathway configure
    delay(10);
  }

  #if SAMPLE_RATE == 88
    while (i2cbb(0x10, 0xbc) != 1) { // clock select 88.2kHz
      delay(10);
    }
  #elif SAMPLE_RATE == 44
    while (i2cbb(0x10, 0xa0) != 1) { // clock select 44.1kHz
      delay(10);
    }
  #elif SAMPLE_RATE == 22
    while (i2cbb(0x10, 0xe0) != 1) { // clock select 22.05kHz
      delay(10);
    }
  #elif SAMPLE_RATE == 8
    while (i2cbb(0x10, 0xac) != 1) { // clock select 8.018kHz
      delay(10);
    }
  #elif SAMPLE_RATE == 2
    while (i2cbb(0x10, 0xce) != 1) { // clock select 2.45kHz
      delay(10);
    }
  #endif

  while (i2cbb(0x12, 0x01) != 1) { // device enable
    delay(10);
  }
  
  // setup ADCs
  #if MOD1 == 0
    DIDR0 = (1 << ADC0D); // turn off digital inputs for ADC0
  #elif (MOD1 == 1)
    DIDR0 = (1 << ADC0D); // turn off digital inputs for ADC0
    ADMUX = 0x00; // start with ADC0 - external VCC for Vref
    ADCSRA = 0xe7; // ADC enable, autotrigger, ck/128
    ADCSRB = 0x00; // free running mode
  #endif
  
  // microdec uses timer0
  // setup timer0 for codec clock division
  TCCR0A = 0x0f; // set to CTC mode, external clock
  TCNT0 = 0x00; // clear the counter
  #if SAMPLE_RATE == 88
    OCR0A = 0x3f; // set the counter top
  #elif (SAMPLE_RATE == 44) || (SAMPLE_RATE == 22)
    OCR0A = 0x7f; // set the counter top
  #endif
  TIMSK0 = 0x02; // turn on compare match interrupt

  sei(); // turn on interrupts
}


// adc sample routine
// this creates relatively low noise 16b values from adc samples
#if (MOD1 == 0)&&(MOD2 == 0)
  static inline void Microdec_mods() {
    // do nothing
  }
#elif (MOD1 == 1)&&(MOD2 == 0)
  static inline void Microdec_mods(unsigned int* _mod1value) {
    if (ADCSRA & (1 << ADIF)) { // check if sample ready
      unsigned int _temp;
      asm volatile ("lds %A0, %1" : "=l" (_temp) : "n" (0x0078) );
      asm volatile ("lds %B0, %1" : "=l" (_temp) : "n" (0x0079) );
      _mod1temp += _temp;
      ADCSRA = 0xf7; // reset the interrupt flag
      if (--_i == 0) { // check if enough samples have been averaged
        // shift value to make a 16b integer
        unsigned char x = 0;
        for (unsigned char t = OVERSAMPLE; t > 1; t >>= 1) {
          x += 1;
        }
        _mod1temp <<= 6 - x;
        // add in hysteresis to remove jitter
        if (((_mod1temp - *_mod1value) < HYST) || ((*_mod1value - _mod1temp) < HYST)) {
        }
        else {
          *_mod1value = _mod1temp; // move temp value
	}
        _mod1temp = 0x0000; // reset temp value
        _i = OVERSAMPLE; // reset counter
      }
    }
  }
#elif (MOD1 == 1)&&(MOD2 == 1)
  static inline void Microdec_mods(unsigned int* _mod1value, unsigned char* _mod2value) {
    if (--_k == 0) {
      unsigned char _mod2temp2 = PINJ; // fetch current pin state
      // check for rising edge on PINJ0
      unsigned char t = (_mod2temp2 ^ _mod2temp) & _mod2temp2 & 0x01;
      if (t) { // sample PINJ1 if true
        if (_mod2temp2 & 0x02) {
          *_mod2value += MOD2INDEX;
        #if (MOD2OVF == 1) // do overflow truncating
          if (*_mod2value < MOD2INDEX) {
            *_mod2value = 0xff;
          }
        #endif
        }
        else {
          *_mod2value -= MOD2INDEX;
        #if (MOD2OVF == 1) // do overflow truncating
          if (*_mod2value > (0xff - MOD2INDEX)) {
            *_mod2value = 0x00;
          }
        #endif
        }
      }
      _mod2temp = _mod2temp2; // store current value for next time
      _k = 64; // reset counter
    }    
    else if (ADCSRA & (1 << ADIF)) { // check if sample ready
      unsigned int _temp;
      asm volatile ("lds %A0, %1" : "=l" (_temp) : "n" _SFR_MEM_ADDR(ADCL) );
      asm volatile ("lds %B0, %1" : "=l" (_temp) : "n" _SFR_MEM_ADDR(ADCH) );
      _mod1temp += _temp;
      ADCSRA = 0xf7; // reset the interrupt flag
      if (--_i == 0) { // check if enough samples have been averaged
        // shift value to make a 16b integer
        unsigned char x = 0;
        for (unsigned char t = OVERSAMPLE; t > 1; t >>= 1) {
          x +=1;
        }
        _mod1temp <<= 6 - x;
        // add in hysteresis to remove jitter
        if (((_mod1temp - *_mod1value) < HYST) || ((*_mod1value - _mod1temp) < HYST)) {
        }
        else {
          *_mod1value = _mod1temp; // move temp value
	}
        _mod1temp = 0x0000; // reset temp value
        _i = OVERSAMPLE; // reset counter
      }
    }
  }
#elif (MOD1 == 0)&&(MOD2 == 1)
  static inline void Microdec_mods(unsigned char* _mod2value) {
    if (--_k == 0) {
      unsigned char _mod2temp2 = PINJ; // fetch current pin state
      // check for rising edge on PINJ0
      unsigned char t = (_mod2temp2 ^ _mod2temp) & _mod2temp2 & 0x01;
      if (t) { // sample PINJ1 if true
        if (_mod2temp2 & 0x02) {
          *_mod2value += MOD2INDEX;
        #if (MOD2OVF == 1) // do overflow truncating
          if (*_mod2value < MOD2INDEX) {
            *_mod2value = 0xff;
          }
        #endif
        }
        else {
          *_mod2value -= MOD2INDEX;
        #if (MOD2OVF == 1) // do overflow truncating
          if (*_mod2value > (0xff - MOD2INDEX)) {
            *_mod2value = 0x00;
          }
        #endif
        }
      }
      _mod2temp = _mod2temp2; // store current value for next time
      _k = 64; // reset counter
    }
  }
#endif


// codec data transfer function
static inline void Microdec_data(int* _lin, int* _rin, int _lout, int _rout) {

  int _out_temp = _lout;
  PORTB |= (1<<PORTB0);  // toggle ss pin (pb0)
  asm volatile ("out %0, %B1" : : "I" (_SFR_IO_ADDR(SPDR)), "r" (_out_temp) );
  PORTB &= ~(1<<PORTB0); // toggle ss pin
  while(!(SPSR & (1<<SPIF))){ // wait for data transfer to complete
  }
  asm volatile ("out %0, %A1" : : "I" (_SFR_IO_ADDR(SPDR)), "r" (_out_temp) );
  asm volatile ("in r3, %0" : : "I" (_SFR_IO_ADDR(SPDR)) );
  _out_temp = _rout;
  while(!(SPSR & (1<<SPIF))){ // wait for data transfer to complete
  }
  asm volatile ("out %0, %B1" : : "I" (_SFR_IO_ADDR(SPDR)), "r" (_out_temp) );
  asm volatile ("in r2, %0" : : "I" (_SFR_IO_ADDR(SPDR)) );
  asm volatile ("movw %0, r2" : "=r" (*_lin) : );
  while(!(SPSR & (1<<SPIF))){ // wait for data transfer to complete
  }
  asm volatile ("out %0, %A1" : : "I" (_SFR_IO_ADDR(SPDR)), "r" (_out_temp) );
  asm volatile ("in r3, %0" : : "I" (_SFR_IO_ADDR(SPDR)) );
  while(!(SPSR & (1<<SPIF))){ // wait for data transfer to complete
  }
  asm volatile ("in r2, %0" : : "I" (_SFR_IO_ADDR(SPDR)) );
  asm volatile ("movw %0, r2" : "=r" (*_rin) : );
}


static inline unsigned char Microdec_mode(void) {
  return (((((PINJ) >> 3) ^ 0xff) + 8) & 0x0f);
}

static inline unsigned char Microdec_button(void) {
  if ((PINJ & (1<<PINJ2)) == 0) {;
    return 1;
  }
  else {
    return 0;
  }
}

#endif // end include guard

