; program: sampler-rotary.asm
; UID = 000002 - this is a unique id so variables dont conflict
; 16b address space with mono data (1.5s sample time)
; sampler - samples while pushbutton is held, plays back sample in a loop
; rotary encoder controlled playback speed

; program overview
;
; data is sent out and taken in from the codec.  the pushbutton is then
; checked to see if recording should occur.  if the button is pressed
; the program records and plays out the current sound.  data is taken
; in on the left channel, and played out on both left and right.  if the
; button is not pressed, the program indexes through the memory and plays
; the next sample.  the speed at which it plays through the memory is
; controlled by the rotary encoder.  turning the encoder to the right
; speeds playback up, and turning it the left slows playback down.  the
; audio is kept clean over fractional sample periods by interpolating
; between the two closest samples.  the volume is brought down for 512
; samples (11.7ms) around the buffer boundary to reduce clicks.

; register usage - may be redefined in other sections
;
; r0  multiply result lsb
; r1  multiply result msb
; r2  left/right lsb out
; r3  left/right msb out
; r4  
; r5  
; r6  left lsb in/multiply accumulation lsb
; r7  left msb in/multiply accumulation msb
; r8  
; r9  
; r10 adc lsb accumulator/encoder0 buffer
; r11 adc msb accumulator/encoder1 buffer
; r12 playback speed increment lsb value ($0100 is normal speed)
; r13 playback speed increment msb value
; r14 rotary encoder counter
; r15 switch counter
; r16 temporary swap register
; r17 temporary swap register
; r18 signed multiply register
; r19 signed multiply register
; r20 ducking state register
; r21 read address fractional byte register
; r22 write address third byte/null register
; r23 ducking counter
; r24 write address lsb
; r25 write address msb
; r26 buffer size lsb (not implemented yet)
; r27 buffer size msb
; r28 read address lsb
; r29 read address msb
; r30 jump location for interrupt lsb
; r31 jump location for interrupt msb

; constant definitions
;
.equ step_size_000002 = $14 ; this is the amount a single detent on the rotary
; encoder changes the read address increment by.  each bit is 1/256 of
; normal playback speed.

; program begins here
; initiate data transfer to codec
sbi portb,portb0 ; toggle slave select pin
out spdr,r3 ; send out left channel msb
cbi portb,portb0
ldi r22,$00 ; set up high byte write register

;for testing - will eventually make a variable buffer size
ldi r27,$ff
ldi r26,$ff

wait1_000002: ; check if byte has been sent

in r17,spsr
sbrs r17,spif
rjmp wait1_000002
in r7,spdr ; recieve in left channel msb
out spdr,r2 ; send out left channel lsb

wait2_000002: ; check if byte has been sent

in r17,spsr
sbrs r17,spif
rjmp wait2_000002
in r6,spdr ; recieve in left channel lsb
out spdr,r3 ; send out right channel msb

wait3_000002: ; check if byte has been sent

in r17,spsr
sbrs r17,spif
rjmp wait3_000002
in r17,spdr ; recieve in right channel msb
out spdr,r2 ; send out right channel lsb

wait4_000002: ; check if byte has been sent

in r17,spsr
sbrs r17,spif
rjmp wait4_000002
in r17,spdr ; recieve in left channel lsb

;check pushbutton
lds r16,pinj ; get pushbutton data
sbrc r16,$02 ; check if button depressed
rjmp interpolate_000002 ; playback if button is not depressed
movw r29:r28,r25:r24 ; synchronize read and write addressses
adiw r25:r24,$01 ; increment write address
;write left channel data to sram
out portd,r24 ; set address
sts porth,r25
out portg,r22 ; pull ce low,we low,and set high bits of address
ldi r17,$ff
out ddra,r17 ; set porta as output for data write
out ddrc,r17 ; set portc as output for data write
out porta,r6 ; set data
out portc,r7
sbi portg,portg2 ; pull we high to write
out ddra,r22 ; set porta as input for data lines
out ddrc,r22 ; set portc as input for data lines
movw r3:r2,r7:r6 ; pass data through while recording
ldi r17,$01 ; set playback speed to normal
mov r12,r22 ; r22 is cleared above
mov r13,r17
rjmp switchsample_000002 ; finish off

interpolate_000002: ; interpolate data based upon speed setting

add r21,r12 ; increment read register
adc r28,r13
adc r29,r22 ; r22 is cleared above
;get left channel sample 1 data from sram
out portd,r28 ; set address
sts porth,r29
nop ; wait input latch time of 2 clock cycles
nop ; wait input latch time of 2 clock cycles
in r6,pina ; get data
in r19,pinc ; get data
adiw r29:r28,$01 ; increment read register
;get left channel sample 2 data from sram
out portd,r28 ; set address
sts porth,r29
nop ; wait input latch time of 2 clock cycles
nop ; wait input latch time of 2 clock cycles
in r7,pina ; get data
in r18,pinc ; get data
sbiw r29:r28,$01 ; reset read register

;multiply sample 1 by distance
mov r16,r21 ; get distance from sample 1
com r16
mulsu r19,r16 ; (signed)ah * b
movw r3:r2,r1:r0
mul	r6,r16	; al * b
add	r2,r1
adc	r3,r22 ; r22 is cleared above
mov r16,r0

;multiply sample 2 by distance
mulsu r18,r21 ; (signed)ah * b
add r2,r0 ; accumulate result
adc r3,r1
mul	r7,r21	; al * b
add r16,r0 ; accumulate result
adc	r2,r1
adc	r3,r22 ; r22 is cleared above

;get fade distance
movw r17:r16,r29:r28 ; move read address to temporary regitser
mov r18,r21
sub r16,r24 ; find distance to loop boundary
sbc r17,r25
brcc half_000002 ; check if result is negative
com r16 ; invert distance if negative
com r17
com r18
ldi r22,$01
add r18,r22
clr r22
adc r16,r22
adc r17,r22

half_000002: ; check if result is greater than half the buffer size

lsr r27 ; divide buffer size by 2
ror r26
cp r16,r26 ; check if result is greater than half the buffer size
cpc r17,r27
brlo reset_000002 ; skip flip if not
lsl r26 ; reset buffer size
rol r27
sub r16,r26 ; flip result around boundary
sbc r17,r27
com r16
com r17
com r18
ldi r22,$01
add r18,r22
clr r22
adc r16,r22
adc r17,r22
rjmp attenuate_000002

reset_000002: ; reset buffer

lsl r26 ; reset buffer size
rol r27

attenuate_000002: ; multiply signal by distance to boundary

;multiply signal if less than ducking time
sbrc r20,$05 ; check if already ducking
rjmp duck_000002
cp r16,r12 ; check if distance is great than $00ff - 256 sample ducking
cpc r17,r13 ; r22 is cleared above
brsh rotary_000002 ; do nothing if not
ldi r20,$20 ; set bit to indicate ducking and downcounting
ser r23 ; initialize ducking counter to top

duck_000002: ; duck the signal

movw r17:r16,r3:r2 ; move data to signed multiply register
sbrc r20,$04 ; check if up or down couting
rjmp holdoff_000002
mulsu r17,r23 ; (signed)ah * b
movw r3:r2,r1:r0
mul	r16,r23 ; al * b
add	r2,r1
adc	r3,r22 ; r22 is cleared above
dec r23
brne rotary_000002 ; change direction if at bottom
ori r20,$10 ; set state to indicate upcounting and bottom holdoff
rjmp rotary_000002

holdoff_000002: ; check for holdoff time

sbrc r20,$03 ; check if holdoff time is done
rjmp upcount_000002 ; mute during holdoff time
clr r2 ; mute output signal
clr r3
inc r20 ; increment holdoff time
rjmp rotary_000002 ; finish off

upcount_000002: ; upcount and multiply

mulsu r17,r23 ; (signed)ah * b
movw r3:r2,r1:r0
mul	r16,r23 ; al * b
add	r2,r1
adc	r3,r22 ; r22 is cleared above
inc r23
brne rotary_000002 ; change direction if at bottom
clr r20 ; change direction back to downcount and indicate ducking done

rotary_000002: ; check rotary encoder and adjust playback rate
; rotary encoder is externally debounced, so that is not done here.
; pin1 is sampled on a transition from high to low on pin0.  if pin1 is
; high, a left turn occured, if pin1 is low, a right turn occured.
dec r14 ; reduce the sampling rate to help with debounce
brne switchsample_000002
ldi r17,$40 ; adjust sample frequency to catch all rising edges (1.5ms)
mov r14,r17
lds r17,pinj ; get switch data
sbrs r17,$00 ; check if pin0 is low
rjmp edge_000002 ; check if pin0 was low on previous sample
clt ;  clear state register if back high
rjmp switchsample_000002 ; finish off

edge_000002: ; check for falling edge

brts switchsample_000002 ; do nothing if the edge was already detected
set ; set state register to indicate a falling edge occured
sbrs r17,$01 ; check if pin1 is high
rjmp increment_000002 ; increment desired delay if right rotation
ldi r17,step_size_000002 ; decrement desired delay register
sub r12,r17
sbc r13,r22 ; r22 is cleared above
rjmp switchsample_000002 ; finish off

increment_000002: ; increment desired delay register

ldi r17,step_size_000002 ; increment desired delay register
add r12,r17
adc r13,r22 ; r22 is cleared above

switchsample_000002: ;check switch

dec r15
brne done_000002
lds r31,pinj ; get switch data
andi r31,$78 ; mask off rotary switch
ldi r17,$02
lsr r31
lsr r31
add r31,r17 ; adjust switch position to program memory location

done_000002:

reti

