; program: sampler-rotary.asm
; UID = 000002 - this is a unique id so variables dont conflict
; 16b address space with mono data (1.5s sample time)
; sampler - samples while pushbutton is held, plays back sample in a loop
; rotary encoder controlled playback speed

; program overview
;
; data is sent out and taken in from the codec.  the pushbutton is then
; checked to see if recording should occur.  if the button is pressed
; the program records and plays out the current sound.  data is taken
; in on the left channel, and played out on both left and right.  if the
; button is not pressed, the program indexes through the memory and plays
; the next sample.  the speed at which it plays through the memory is
; controlled by the rotary encoder.  turning the encoder to the right
; slows playback down, and turning it the left speeds playback up.  the
; audio is kept clean over fractional sample periods by interpolating
; between the two closest samples.

; register usage - may be redefined in other sections
;
; r0  multiply result lsb
; r1  multiply result msb
; r2  left/right lsb out
; r3  left/right msb out
; r4  rotary encoder pin0 debounce register
; r5  rotary encoder pin1 debounce register
; r6  left lsb in/multiply accumulation lsb
; r7  left msb in/multiply accumulation msb
; r8  multiply acccumulation fractional register
; r9  pushbutton debounce register
; r10 adc lsb accumulator/encoder0 buffer
; r11 adc msb accumulator/encoder1 buffer
; r12 playback speed increment lsb value ($0100 is normal speed)
; r13 playback speed increment msb value
; r14 
; r15 switch counter
; r16 temporary swap register
; r17 temporary swap register
; r18 signed multiply register
; r19 signed multiply register
; r20 
; r21 read address fractional byte register
; r22 write address third byte/null register
; r23 read address third byte
; r24 write address lsb
; r25 write address msb
; r26 
; r27 rotary encoder previous state register
; r28 read address lsb
; r29 read address msb
; r30 jump location for interrupt lsb
; r31 jump location for interrupt msb

; constant definitions
;
.equ step_size_000002 = $14 ; this is the amount a single detent on the rotary
; encoder changes the read address increment by.  each bit is 1/256 of
; normal playback speed.

; program begins here
; initiate data transfer to codec
sbi portb,portb5 ; toggle slave select pin
out spdr,r3 ; send out left channel msb
cbi portb,portb5
ldi r23,$04 ; set up high byte read register
ldi r22,$00 ; set up high byte write register

wait1_000002: ; check if byte has been sent

in r17,spsr
sbrs r17,spif
rjmp wait1_000002
in r7,spdr ; recieve in left channel msb
out spdr,r2 ; send out left channel lsb

wait2_000002: ; check if byte has been sent

in r17,spsr
sbrs r17,spif
rjmp wait2_000002
in r6,spdr ; recieve in left channel lsb
out spdr,r3 ; send out right channel msb

wait3_000002: ; check if byte has been sent

in r17,spsr
sbrs r17,spif
rjmp wait3_000002
in r17,spdr ; recieve in right channel msb
out spdr,r2 ; send out right channel lsb

wait4_000002: ; check if byte has been sent

in r17,spsr
sbrs r17,spif
rjmp wait4_000002
in r17,spdr ; recieve in left channel lsb

;check pushbutton
lds r16,pinj ; get pushbutton data
bst r16,$02 ; debounce pushbutton
lsl r9
bld r9,$00
ldi r17,$ff ; check if button not pressed
cp r9,r17
breq interpolate_000002 ; playback if button is not depressed
movw r29:r28,r25:r24 ; synchronize read and write addressses
adiw r25:r24,$01 ; increment write address
;write left channel data to sram
out portd,r24 ; set address
sts porth,r25
out portg,r22 ; pull ce low,we low,and set high bits of address
ldi r17,$ff
out ddra,r17 ; set porta as output for data write
out ddrc,r17 ; set portc as output for data write
out porta,r6 ; set data
out portc,r7
ldi r17,$00 ; prepare for setting ports a,c as input
sbi portg,portg2 ; pull we high to write
out ddra,r17 ; set porta as input for data lines
out ddrc,r17 ; set portc as input for data lines
movw r3:r2,r7:r6 ; pass data through while recording
ldi r17,$01 ; set playback speed to normal
mov r12,r22 ; r22 is cleared above
mov r13,r17

switchsample_000002: ;check switch

dec r15
brne done_000002
lds r31,pinj ; get switch data
andi r31,$78 ; mask off rotary switch
ldi r17,$02
lsr r31
lsr r31
add r31,r17 ; adjust switch position to program memory location

done_000002:

reti

interpolate_000002: ; interpolate data based upon speed setting

add r21,r12 ; increment read register
adc r28,r13
adc r29,r22 ; r22 is cleared above
;get left channel sample 1 data from sram
out portd,r28 ; set address
sts porth,r29
nop ; wait input latch time of 2 clock cycles
nop ; wait input latch time of 2 clock cycles
in r2,pina ; get data
in r19,pinc ; get data
adiw r29:r28,$01 ; increment read register
;get left channel sample 2 data from sram
out portd,r28 ; set address
sts porth,r29
nop ; wait input latch time of 2 clock cycles
nop ; wait input latch time of 2 clock cycles
in r3,pina ; get data
in r18,pinc ; get data
sbiw r29:r28,$01 ; reset read register

;multiply sample 1 by distance
mov r16,r21 ; get distance from sample 1
com r16
mulsu r19,r16 ; (signed)ah * b
movw r7:r6,r1:r0
mul	r2,r16	; al * b
add	r6,r1
adc	r7,r22 ; r22 is cleared above
mov r8,r0

;multiply sample 2 by distance
mulsu r18,r21 ; (signed)ah * b
add r6,r0 ; accumulate result
adc r7,r1
mul	r3,r21	; al * b
add r8,r0 ; accumulate result
adc	r6,r1
adc	r7,r22 ; r22 is cleared above
movw r3:r2,r7:r6 ; move result to ouput registers

tst r15
brne switchsample_000002
lds r17,pinj ; get switch data
bst r17,$00 ; debounce pin0 of encoder
lsl r4
bld r4,$00
bst r17,$01 ; debounce pin1 of encoder
lsl r5
bld r5,$00
tst r5 ; see if pin1 is low
breq edge_000002
ldi r17,$ff ; check if fully high
cp r5,r17
brne switchsample_000002 ; do nothing if not fully high
ldi r27,$01 ; set previous state to current state
rjmp switchsample_000002

edge_000002: ; check for falling edge

sbrs r27,$00 ; check if previous state was high
rjmp switchsample_000002 ; do nothing if no state change
ldi r27,$00 ; clear register
sbrc r4,$00 ; check average state of pin1
inc r27
sbrc r4,$01
inc r27
sbrc r4,$02
inc r27
sbrc r4,$03
inc r27
sbrc r4,$04
inc r27
sbrc r4,$05
inc r27
sbrc r4,$06
inc r27
sbrc r4,$07
inc r27
cpi r27,$04 ; check average value of pin1
ldi r27,$00 ; reset previous state register
brlt increment_000002 ; increment if a backwards rotation
ldi r17,step_size_000002 ; decrement read register
sub r12,r17
sbc r13,r22 ; r22 is cleared above
rjmp switchsample_000002

increment_000002:

ldi r17,step_size_000002 ; increment read register
add r12,r17
adc r13,r22 ; r22 is cleared above
rjmp switchsample_000002 ; finish off

