; program: pitch_shifter-16b-variable-ducking.asm
; UID = 000022 - this is a unique id so variables dont conflict
; 16b address space with mono data (1.5s sample time)
; pitch shifter - loops past n seconds at a higher or lower rate to change
; the apparent pitch of the sound - rotary encoder controlled playback speed
; - pot controlled loop time - ducking around loop boundary

; program overview
;
; data is sent out and taken in from the codec.  data is taken in on the
; left channel, and played out on both left and right.  a buffer of the
; past n seconds is kept and the output is the result of sampling this
; buffer at varying playback speeds. the speed at which it plays through
; the memory is controlled by the rotary encoder.  turning the encoder to
; the left slows playback down, and turning it the right speeds playback
; up.  the audio is kept clean over fractional sample periods by
; interpolating between the two closest samples.  the potentiometer
; controls the total loop time.  it is averaged over 256 samples and
; shifted to a 16b number.  when playback crosses the loop boundary, it
; is attenuated to eliminate clicks.  this is kept at a constant time
; interval regardless of pitch shift amount.

; register usage - may be redefined in other sections
;
; r0  multiply result lsb
; r1  multiply result msb
; r2  adc accumulate lsb
; r3  adc accumulate msb
; r4  left/right lsb out
; r5  left/right msb out
; r6  left lsb in/multiply accumulation lsb
; r7  left msb in/multiply accumulation msb
; r8  multiply acccumulation fractional register
; r9  adc accumulate fractional byte
; r10 encoder0 buffer
; r11 encoder1 buffer
; r12 playback speed increment lsb value ($0100 is normal speed)
; r13 playback speed increment msb value
; r14 encoder state register
; r15 switch counter
; r16 temporary swap register
; r17 temporary swap register
; r18 signed multiply register
; r19 signed multiply register
; r20 ducking counter
; r21 ducking state register
; r22 write address third byte/null register
; r23 read address fractional byte
; r24 write address lsb
; r25 write address msb
; r26 buffer size lsb
; r27 buffer size msb
; r28 read address lsb
; r29 read address msb
; r30 jump location for interrupt lsb
; r31 jump location for interrupt msb

; constant definitions
;
.equ step_size_000022 = $02 ; this is the amount a single detent on the rotary
; encoder changes the read address increment by.  each bit is 1/256 of
; normal playback speed.

; program starts here first time
; intialize registers
ldi r30,$04 ; increment z pointer to new jump location
ldi r17,$01 ; initialize playback speed to normal
mov r13,r17
clr r12

; program begins here every time but first
; initiate data transfer to codec
sbi portb,portb5 ; toggle slave select pin
out spdr,r5 ; send out left channel msb
cbi portb,portb5
ldi r22,$00 ; set up high byte write register

wait1_000022: ; check if byte has been sent

in r17,spsr
sbrs r17,spif
rjmp wait1_000022
in r7,spdr ; recieve in left channel msb
out spdr,r4 ; send out left channel lsb

wait2_000022: ; check if byte has been sent

in r17,spsr
sbrs r17,spif
rjmp wait2_000022
in r6,spdr ; recieve in left channel lsb
out spdr,r5 ; send out right channel msb
;write left channel data to sram
out portd,r24 ; set address
sts porth,r25
out portg,r22 ; pull ce low,we low,and set high bits of address
ldi r17,$ff
out ddra,r17 ; set porta as output for data write
out ddrc,r17 ; set portc as output for data write
out porta,r6 ; set data
out portc,r7
sbi portg,portg2 ; pull we high to write
out ddra,r22 ; set porta as input for data lines
out ddrc,r22 ; set portc as input for data lines

wait3_000022: ; check if byte has been sent

in r17,spsr
sbrs r17,spif
rjmp wait3_000022
in r17,spdr ; recieve in right channel msb
out spdr,r4 ; send out right channel lsb

wait4_000022: ; check if byte has been sent

in r17,spsr
sbrs r17,spif
rjmp wait4_000022
in r17,spdr ; recieve in right channel lsb

;increment write addresses
adiw r25:r24,$01 ; increment write address
cp r24,r26 ; check if at end of buffer
cpc r25,r27
brlo interpolate_000022 ; do nothing if not at end of buffer
clr r24 ; reset buffer to bottom
clr r25

interpolate_000022: ; interpolate data based upon speed setting

add r23,r12 ; increment read register
adc r28,r13
adc r29,r22 ; r22 is cleared above
cp r28,r26 ; check if at end of buffer
cpc r29,r27
brlo getsample1_000022 ; do nothing if not at end of buffer
clr r28 ; reset buffer to bottom
clr r29

getsample1_000022: ;get left channel sample 1 data from sram

out portd,r28 ; set address
sts porth,r29
nop ; wait input latch time of 2 clock cycles
nop ; wait input latch time of 2 clock cycles
in r6,pina ; get data
in r17,pinc ; get data
adiw r29:r28,$01 ; increment read register
cp r28,r26 ; check if at end of buffer
cpc r29,r27
brlo getsample2_000022 ; do nothing if not at end of buffer
clr r28 ; reset buffer to bottom
clr r29

getsample2_000022: ;get left channel sample 2 data from sram

out portd,r28 ; set address
sts porth,r29
nop ; wait input latch time of 2 clock cycles
nop ; wait input latch time of 2 clock cycles
in r7,pina ; get data
in r18,pinc ; get data
cp r28,r22 ; check if at buffer boundary
cpc r29,r22 ; r22 is cleared above
brne interpolate2_000022 ; do nothing if not
movw r29:r28,r27:r26 ; move boundary back to read address

interpolate2_000022: ; multiply sample 1 by distance

sbiw r29:r28,$01 ; reset read register
mov r16,r23 ; get distance from sample 1
com r16
mulsu r17,r16 ; (signed)ah * b
movw r5:r4,r1:r0
mul	r6,r16	; al * b
add	r4,r1
adc	r5,r22 ; r22 is cleared above
mov r8,r0

;multiply sample 2 by distance
mulsu r18,r23 ; (signed)ah * b
add r4,r0 ; accumulate result
adc r5,r1
mul	r7,r23	; al * b
add r8,r0 ; accumulate result
adc	r4,r1
adc	r5,r22 ; r22 is cleared above

;get fade distance
movw r17:r16,r29:r28 ; move read address to temporary regitser
mov r18,r23
sub r16,r24 ; find distance to loop boundary
sbc r17,r25
brcc half_000022 ; check if result is negative
com r16 ; invert distance if negative
com r17
com r18
ldi r22,$01
add r18,r22
clr r22
adc r16,r22
adc r17,r22

half_000022: ; check if result is greater than half the buffer size

lsr r27 ; divide buffer size by 2
ror r26
cp r16,r26 ; check if result is greater than half the buffer size
cpc r17,r27
brlo reset_000022 ; skip flip if not
lsl r26 ; reset buffer size
rol r27
sub r16,r26 ; flip result around boundary
sbc r17,r27
com r16
com r17
com r18
ldi r22,$01
add r18,r22
clr r22
adc r16,r22
adc r17,r22
rjmp attenuate_000022

reset_000022: ; reset buffer

lsl r26 ; reset buffer size
rol r27

attenuate_000022: ; multiply signal by distance to boundary

;multiply signal if less than ducking time
sbrc r21,$01 ; check if already ducking
rjmp duck_000022
tst r13 ; check if going slower or faster than normal speed
brne fastcheck_000022 ; subtract $0100 if going faster
mov r18,r12 ; move to temporary register
neg r18 ; subtract from $0100 if going slower
cp r16,r18 ; number of samples till boundary is proportional to sample index
cpc r17,r13 ; check if distance is great than $00ff - 256 sample ducking
brsh rotary_000022 ; do nothing if not
ldi r21,$02 ; set bit to indicate ducking and downcounting
ser r20 ; initialize ducking counter to top
rjmp duck_000022

fastcheck_000022: ; subtract $0100 if going faster

mov r18,r13 ; move to temporary register
subi r18,$01 ; subtract $0100
cp r16,r12 ; number of samples till boundary is proportional to sample index
cpc r17,r18 ; check if distance is great than $00ff - 256 sample ducking
brsh rotary_000022 ; do nothing if not
ldi r21,$02 ; set bit to indicate ducking and downcounting
ser r20 ; initialize ducking counter to top

duck_000022: ; duck the signal

movw r17:r16,r5:r4 ; move data to signed multiply register
sbrc r21,$00 ; check if up or down couting
rjmp upcount_000022
mulsu r17,r20 ; (signed)ah * b
movw r5:r4,r1:r0
mul	r16,r20 ; al * b
add	r4,r1
adc	r5,r22 ; r22 is cleared above
dec r20
brne rotary_000022 ; change direction if at bottom
ori r21,$01 ; set r21 to indicate upcounting
rjmp rotary_000022

upcount_000022: ; upcount and multiply

mulsu r17,r20 ; (signed)ah * b
movw r5:r4,r1:r0
mul	r16,r20 ; al * b
add	r4,r1
adc	r5,r22 ; r22 is cleared above
inc r20
brne rotary_000022 ; change direction if at bottom
clr r21 ; change direction back to downcount and indicate ducking done

rotary_000022: ; check rotary encoder

dec r15
brne switchsample_000022
lds r17,pinj ; get switch data
bst r17,$00 ; debounce pin0 of encoder
lsl r10
bld r10,$00
bst r17,$01 ; debounce pin1 of encoder
lsl r11
bld r11,$00
tst r11 ; see if pin1 is low
breq edge_000022
ldi r17,$ff ; check if fully high
cp r11,r17
brne switchsample_000022 ; do nothing if not fully high
ldi r16,$01 ; set previous state to current state
mov r14,r16
rjmp switchsample_000022

edge_000022: ; check for falling edge

sbrs r14,$00 ; check if previous state was high
rjmp switchsample_000022 ; do nothing if no state change
clr r14 ; clear register
sbrc r10,$00 ; check average state of pin1
inc r14
sbrc r10,$01
inc r14
sbrc r10,$02
inc r14
sbrc r10,$03
inc r14
sbrc r10,$04
inc r14
sbrc r10,$05
inc r14
sbrc r10,$06
inc r14
sbrc r10,$07
inc r14
mov r16,r14
cpi r16,$04 ; check average value of pin1
clr r14 ; reset previous state register
brlo decrement_000022 ; decrement if a backwards rotation
ldi r17,step_size_000022 ; increment read register
add r12,r17
adc r13,r22 ; r22 is cleared above
rjmp switchsample_000022

decrement_000022:

ldi r17,step_size_000022 ; decrement read register
sub r12,r17
sbc r13,r22 ; r22 is cleared above

switchsample_000022: ;check switch

tst r15
brne adcsample_000022
lds r16,pinj ; get switch data
andi r16,$78 ; mask off rotary switch
ldi r17,$02
lsr r16
lsr r16
add r16,r17 ; adjust switch position to program memory location
cpse r16,r31 ; check if location has changed
clr r30 ; reset jump register to intial state
mov r31,r16
ldi r17,$20 ; speed up switch sampling for better resolution on rotary encoder
mov r15,r17

adcsample_000022: ; get loop setting

lds r17,adcsra ; get adc control register
sbrs r17,adif ; check if adc conversion is complete
rjmp done_000022 ; skip adc sampling
lds r17,adcl ; get low byte adc value
lds r16,adch ; get high byte adc value
add r2,r17
adc r3,r16 ; accumulate adc samples
adc r9,r22 ; accumulate adc samples - r22 is cleared above
ldi r17,$f7
sts adcsra,r17 ; clear interrupt flag
dec r19 ; countdown adc sample clock
brne done_000022 ; move adc value to loop setting after 256 samples
lsr r3 ; multiply accumulated value by 64
ror r2
lsr r3
ror r2
ldi r17,$c0 ; mask off less than 12b
and r2,r17
movw r17:r16,r27:r26 ; make a copy of current loop time for comparison
sub r16,r2 ; find difference between current loop time and last loop time
sbc r17,r3
brcc hysteresis_000022 ; see if difference is large enough to indicate a change
neg r16 ; invert difference if negative
adc r17,r22 ; r22 is cleared above
neg r17

hysteresis_000022: ; see if pot has moved or if its just noise

cpi r16,$c0 ; see if difference is greater than 2 lsb
cpc r17,r22 ; r22 is cleared above
brlo nochange_000022 ; dont update loop time if difference is not large enough
movw r27:r26,r3:r2 ; move adc value to loop time register

nochange_000022: ; clear accumulation registers

clr r2 ; empty accumulation registers
clr r3
clr r9

done_000022:

reti
