; program: pitch_shifter-16b-rotary.asm
; UID = 000019 - this is a unique id so variables dont conflict
; 16b address space with mono data (1.5s sample time)
; pitch shifter - loops past n seconds at a higher or lower rate to change
; the apparent pitch of the sound - rotary encoder controlled playback speed

; program overview
;
; data is sent out and taken in from the codec.  data is taken in on the
; left channel, and played out on both left and right.  a buffer of the
; past n seconds is kept and the output is the result of sampling this
; buffer at varying playback speeds. the speed at which it plays through
; the memory is controlled by the rotary encoder.  turning the encoder to
; the right slows playback down, and turning it the left speeds playback up.  the
; audio is kept clean over fractional sample periods by interpolating
; between the two closest samples.

; register usage - may be redefined in other sections
;
; r0  multiply result lsb
; r1  multiply result msb
; r2  left/right lsb out
; r3  left/right msb out
; r4  rotary encoder pin0 debounce register
; r5  rotary encoder pin1 debounce register
; r6  left lsb in/multiply accumulation lsb
; r7  left msb in/multiply accumulation msb
; r8  multiply acccumulation fractional register
; r9  pushbutton debounce register
; r10 adc lsb accumulator/encoder0 buffer
; r11 adc msb accumulator/encoder1 buffer
; r12 playback speed increment lsb value ($0100 is normal speed)
; r13 playback speed increment msb value
; r14 
; r15 switch counter
; r16 temporary swap register
; r17 temporary swap register
; r18 signed multiply register
; r19 signed multiply register
; r20 unsigned multiply register
; r21 unsigned multiply register
; r22 write address third byte/null register
; r23 read address third byte
; r24 write address lsb
; r25 write address msb
; r26 
; r27 
; r28 read address lsb
; r29 read address msb
; r30 jump location for interrupt lsb
; r31 jump location for interrupt msb

; constant definitions
;
.equ step_size_000019 = $44 ; this is the amount a single detent on the rotary
; encoder changes the read address increment by.  each bit is 1/256 of
; normal playback speed.

; program begins here
; initiate data transfer to codec
sbi portb,portb5 ; toggle slave select pin
out spdr,r5 ; send out left channel msb
cbi portb,portb5
ldi r23,$04 ; set up high byte read register
ldi r22,$00 ; set up high byte write register

wait1_000019: ; check if byte has been sent

in r17,spsr
sbrs r17,spif
rjmp wait1_000019
in r7,spdr ; recieve in left channel msb
out spdr,r4 ; send out left channel lsb

wait2_000019: ; check if byte has been sent

in r17,spsr
sbrs r17,spif
rjmp wait2_000019
in r6,spdr ; recieve in left channel lsb
out spdr,r5 ; send out right channel msb
;write left channel data to sram
out portd,r24 ; set address
sts porth,r25
out portg,r22 ; pull ce low,we low,and set high bits of address
ldi r17,$ff
out ddra,r17 ; set porta as output for data write
out ddrc,r17 ; set portc as output for data write
out porta,r6 ; set data
out portc,r7
sbi portg,portg2 ; pull we high to write
out ddra,r22 ; set porta as input for data lines
out ddrc,r22 ; set portc as input for data lines

wait3_000019: ; check if byte has been sent

in r17,spsr
sbrs r17,spif
rjmp wait3_000019
in r17,spdr ; recieve in right channel msb
out spdr,r4 ; send out right channel lsb
;get right channel data from sram
out portd,r28 ; set address
sts porth,r29
nop ; wait input latch time of 2 clock cycles
nop ; wait input latch time of 2 clock cycles
in r18,pina ; get data
in r19,pinc ; get data

wait4_000019: ; check if byte has been sent

in r17,spsr
sbrs r17,spif
rjmp wait4_000019
in r17,spdr ; recieve in right channel lsb

;increment read and write addresses
adiw r25:r24,$01 ; increment write address
cp r24,r26 ; check if at end of buffer
cpc r25,r27
brlo readinc_000019 ; do nothing if not at end of buffer
clr r24 ; reset buffer to bottom
clr r25

readinc_000019: ; increment read address

adiw r29:r28,$02 ; increment read address
cp r28,r26 ; check if at end of buffer
cpc r29,r27
brlo fade_000019 ; do nothing if not at end of buffer
clr r28 ; reset buffer to bottom
clr r29

fade_000019: ; get fade distance

movw r21:r20,r29:r28 ; move read address to temporary register
sub r20,r24 ; find distance to loop boundary
sbc r21,r25
brcc half_000019 ; check if result is negative
neg r20 ; invert distance if negative
adc r21,r22 ; r22 is cleared above
neg r21

half_000019: ; check if result is greater than half the buffer size

lsr r27 ; divide buffer size by 2
ror r26
cp r20,r26 ; check if result is greater than half the buffer size
cpc r21,r27
brlo reset_000019 ; skip flip if not
lsl r26 ; reset buffer size
rol r27
sub r20,r26 ; flip result around boundary
sbc r21,r27
neg r20
adc r21,r22 ; r22 is cleared above
neg r21
rjmp scale_000019

reset_000019: ; reset buffer

lsl r26 ; reset buffer size
rol r27

scale_000019: ; create scaling factor - only good to 0.5 - 1.0 accuracy

movw r17:r16,r27:r26 ; move buffer size to temporary register
sec ; set carry flag in case of r27:r26 = 0

scale1_000019: ; iterate scaling factor

rol r16 ; check for highest bit
rol r17
brcs attenuate_000019 ; if highest bit is set, stop scaling
lsl r20 ; scale up buffer distance
rol r21
rjmp scale1_000019 ; continue scaling

attenuate_000019: ; multiply signal by distance to boundary

ror r17 ; move buffer size back into max range
ror r16 ; will be used in next section
;multiply signal
muls r19,r21 ; (signed)ah * (signed)bh
movw r5:r4,r1:r0
mul	r18,r20	; al * bl
movw r3:r2,r1:r0
mulsu r19,r20 ; (signed)ah * bl
sbc	r5,r22 ; r22 is cleared above
add	r3,r0
adc	r4,r1
adc	r5,r22
mulsu r21,r18 ; (signed)bh * al
sbc	r5,r22 ; r22 is cleared above
add	r3,r0
adc	r4,r1
adc	r5,r22

;get sample from opposite side of buffer
movw r19:r18,r29:r28 ; move read address to temporary register
lsr r27 ; half buffer size
ror r26
sub r18,r26 ; move read address halfway back in buffer
sbc r19,r27
brcc reset1_000019 ; skip if no underflow
lsl r26 ; reset buffer size
rol r27
add r18,r26 ; move read address to other side of buffer if underflow
adc r19,r27
rjmp getdata_000019 ; get data

reset1_000019: ; get data from memory

lsl r26 ; reset buffer size
rol r27

getdata_000019: ; get data from memory

out portd,r18 ; set address
sts porth,r19
nop ; wait setup time
nop
in r18,pina ; get data
in r19,pinc ; get data

;distance is the opposite of previous result
lsr r17 ; half the scaled buffer size
ror r16
sub r16,r20 ; take the distance to half of the buffer size
sbc r17,r21
movw r21:r20,r17:r16 ;  move result to multiply register
;multiply and accumulate with result from above - r5:r4:r3:r2
muls r19,r21 ; (signed)ah * (signed)bh
add	r4,r0
adc	r5,r1
mul	r18,r20 ; al * bl
add	r2,r0
adc	r3,r1
adc	r4,r22
adc	r5,r22
mulsu r19,r20 ; (signed)ah * bl
sbc	r5,r22
add	r3,r0
adc	r4,r1
adc	r5,r22
mulsu r21,r18 ; (signed)bh * al
sbc	r5,r22
add	r3,r0
adc	r4,r1
adc	r5,r22
rjmp rotary_000019

switchsample_000019: ;check switch

tst r15
brne done_000019
lds r31,pinj ; get switch data
andi r31,$78 ; mask off rotary switch
ldi r17,$02
lsr r31
lsr r31
add r31,r17 ; adjust switch position to program memory location
ldi r17,$20 ; speed up switch sampling for better resolution on rotary encoder
mov r15,r17

done_000019:

reti

interpolate_000019: ; interpolate data based upon speed setting

add r21,r12 ; increment read register
adc r28,r13
adc r29,r22 ; r22 is cleared above
;get left channel sample 1 data from sram
out portd,r28 ; set address
sts porth,r29
nop ; wait input latch time of 2 clock cycles
nop ; wait input latch time of 2 clock cycles
in r2,pina ; get data
in r19,pinc ; get data
adiw r29:r28,$01 ; increment read register
;get left channel sample 2 data from sram
out portd,r28 ; set address
sts porth,r29
nop ; wait input latch time of 2 clock cycles
nop ; wait input latch time of 2 clock cycles
in r3,pina ; get data
in r18,pinc ; get data
sbiw r29:r28,$01 ; reset read register

;multiply sample 1 by distance
mov r16,r21 ; get distance from sample 1
com r16
mulsu r19,r16 ; (signed)ah * b
movw r7:r6,r1:r0
mul	r2,r16	; al * b
add	r6,r1
adc	r7,r22 ; r22 is cleared above
mov r8,r0

;multiply sample 2 by distance
mulsu r18,r21 ; (signed)ah * b
add r6,r0 ; accumulate result
adc r7,r1
mul	r3,r21	; al * b
add r8,r0 ; accumulate result
adc	r6,r1
adc	r7,r22 ; r22 is cleared above
movw r3:r2,r7:r6 ; move result to ouput registers

rotary_000019: ; check rotary encoder

dec r15
brne switchsample_000019
lds r17,pinj ; get switch data
bst r17,$00 ; debounce pin0 of encoder
lsl r10
bld r10,$00
bst r17,$01 ; debounce pin1 of encoder
lsl r11
bld r11,$00
tst r11 ; see if pin1 is low
breq edge_000019
ldi r17,$ff ; check if fully high
cp r11,r17
brne switchsample_000019 ; do nothing if not fully high
ldi r16,$01 ; set previous state to current state
mov r14,r16
rjmp switchsample_000019

edge_000019: ; check for falling edge

sbrs r14,$00 ; check if previous state was high
rjmp switchsample_000019 ; do nothing if no state change
clr r14 ; clear register
sbrc r10,$00 ; check average state of pin1
inc r14
sbrc r10,$01
inc r14
sbrc r10,$02
inc r14
sbrc r10,$03
inc r14
sbrc r10,$04
inc r14
sbrc r10,$05
inc r14
sbrc r10,$06
inc r14
sbrc r10,$07
inc r14
mov r16,r14
cpi r16,$04 ; check average value of pin1
clr r14 ; reset previous state register
brlo increment_000019 ; increment if a backwards rotation
ldi r17,step_size_000019 ; decrement read register
sub r26,r17
sbc r27,r22 ; r22 is cleared above
rjmp switchsample_000019

increment_000019:

ldi r17,step_size_000019 ; increment read register
add r26,r17
adc r27,r22 ; r22 is cleared above
rjmp switchsample_000019 ; finish off


